import { FastifyReply, FastifyRequest } from "fastify";
import { exchangeModel } from "../models/exchange";
import { BaseService } from "./base";
import { exchangeValidations } from "../validations/exchange";
import { referenceModel } from "../models/reference";
import prisma from "../lib/prisma";
import { sendSms } from "./sms";
import { userModel } from "../models/user";
import { Usuario } from "@prisma/client";
import { mailService } from "./mail";
import { benefitModel } from "../models/benefit";
import { smsCompleteExchange } from "../templates/sms/sms";
import { categoryModel } from "../models/category";

class ExchangeService extends BaseService {
  model = exchangeModel;

  create = async (req: FastifyRequest, res: FastifyReply) => {
    const data = exchangeValidations.getData.parse(req.body);

    const reference = await referenceModel.getUnusedReferences(
      data.id_beneficio
    );
    if (!reference) {
      throw new Error("Não há referências disponíveis para este benefício.");
    }

    const user = await userModel.getById(data.id_usuario);

    const { id_beneficio, ...exchangeData } = data;

    const benefit = await benefitModel.getById(id_beneficio);
    const category = await categoryModel.getById(benefit?.id_categoria!);

    const operation = await prisma.$transaction(async (tx) => {
      const exchange = await tx.troca.create({
        data: {
          ...exchangeData,
          id_referencia: reference.id,
        },
      });

      await tx.referencia.update({
        where: { id: reference.id },
        data: { usado: true },
      });

      return exchange;
    });

    const createdAt = new Date().toISOString();

    if (data.telefone)
      await sendSms(data.telefone, smsCompleteExchange(reference.codigo));
    if (data.email) {
      await mailService.sendCompleteExchange(data.email, {
        name: user?.nome!,
        code: reference.codigo,
        created_at: createdAt,
        value: benefit?.pontos.toString()!,
        benefit: benefit?.nome!,
        category: category?.nome!,
        description: benefit?.descricao!,
      });
    }

    return res.send(operation);
  };
}

export const exchangeService = new ExchangeService();
